<?php

namespace App\Http\Controllers;

use App\Models\CoffeeParcelRegistry;
use App\Models\Georeferencing;
use App\Models\Producer;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use App\Services\ImportadorGeoreferenciaciones;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Cache;
use Maatwebsite\Excel\Facades\Excel;

class GeoreferencingController extends Controller
{
    private $koboApiToken;
    private $geoAssetUid;
    private $parcelaAssetUid;
    private $padronApiUid;
    private $koboBaseUrl;
    
    // Cache por 10 segundos (ajustable - para producción usar 3600)
    private $cacheDuration = 10;

    public function __construct()
    {
        // Obtener configuración desde config/services.php
        $koboConfig = config('services.kobotoolbox');
        
        $this->koboApiToken = $koboConfig['KOBO_API_TOKEN'];
        $this->geoAssetUid = $koboConfig['GEO_ASSET_UID'];
        $this->parcelaAssetUid = $koboConfig['PARCELA_ASSET_UID'];
        $this->padronApiUid = $koboConfig['PADRON_ASSET_UID'];
        $this->koboBaseUrl = $koboConfig['server_url'];
    }

    public function index()
    {
        $currentYear = date('Y');
        $TotalPoligono = Georeferencing::all()->count();

        $PoligonosActivos = Georeferencing::whereHas('parcela', function($query){
            $query->whereHas('productor', function($q){
                $q->where('activo', 1); //solo productores activos
            });
        })->count();

        $PoligonoNoActivos = Georeferencing::whereHas('parcela', function($query){
            $query->whereHas('productor', function($q){
                $q->where('activo', 0); //solo productores no activos
            });
        })->count();



        // $ProductoresConPoligono = Producer::where('activo', 1)
        //     ->whereHas('parcels.georefrenciacion') // relación anidada
        //     ->pluck('id_productor');
        $parcelasActivas = CoffeeParcelRegistry::whereHas('productor', function($q) {
            $q->where('activo', 1);
        })->pluck('id_parcela');

        $parcelasConPoligono = Georeferencing::whereHas('parcela', function($q) {
            $q->whereHas('productor', function($q2){
                $q2->where('activo', 1);
            });
        })->pluck('codigo_parcela');

        $parcelasSinPoligono = $parcelasActivas->diff($parcelasConPoligono)->count();

        // $detalleParcelasSinPoligono = CoffeeParcelRegistry::whereIn('id_parcela', $parcelasSinPoligono)
        //     ->with('productor') // para ver también el productor
        //     ->get();


        $ProductoresSinPoligono = Producer::where('activo', 1)
            ->whereDoesntHave('parcelas.georefrenciacion') // no tienen georeferenciación
            ->count();


        return view('Georeferenciacion.index', compact('TotalPoligono', 'PoligonosActivos', 'ProductoresSinPoligono', 'parcelasSinPoligono', 'PoligonoNoActivos'));
    }

    public function importarKobo(Request $request)
    {
        $validator = Validator::make($request->all(),[
            'year' => 'required|integer|min:2020|max:' . (date('Y') + 1),
            'auth_method' => 'required|in:basic,token',
            'auth_credential' => 'required|string',
            'second_auth_credential' => 'required_if:auth_method,basic|nullable|string',
            'asset_id' => 'required|string',
            'server_url' => 'required|url'
        ]);

        if ($validator->fails()){
            return response()->json([
                'success' => false,
                'message' => 'Datos de entrada inválidos',
                'errors' => $validator->errors()
            ], 422);
        }

        try{
            $data = $validator->validated();

            //Crear instancia del importador
            $importador = new ImportadorGeoreferenciaciones(
                year: $data['year'],
                assetId: $data['asset_id'],
                username: $data['auth_credential'],
                password: $data['second_auth_credential'] ?? null,
                serverUrl: $data['server_url']
            );



            //Ejecutamos la importación
            $resultado = $importador->importar();



            if ($resultado['success']){
                Log::info('Importación exitosa desde KoBoToolbox', [
                    'year' => $data['year'],
                    'total_registros' => $resultado['total']
                ]);
            }
            return response()->json($resultado);

        }catch(\Exception $e){
            Log::error('Error en importación KoBoToolbox',[
                'error' => $e->getMessage(),
                'year' => $request->year
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Error interno del servidor: ' . $e->getMessage()
            ], 500);
        }
    }

    // Obtenemos información del asset antes de importar
    public function infoAsset (Request $request)
    {
        $validator = Validator::make($request->all(),[
            'auth_method' => 'required|in:basic,token',
            'auth_credential' => 'required|string',
            'second_auth_credential' => 'required_if:auth_method,basic|nullable|string',
            'asset_id' => 'required|string',
            'server_url' => 'required|url',
            'year' => 'required|integer|min:2020'
        ]);

        if ($validator->fails()){
            return response()->json([
                'success' => false,
                'message' => 'Datos de entrada inválidos',
                'error' => $validator->errors()
            ], 422);
        }



        try{
            $data = $validator->validated();

            $importador = new ImportadorGeoreferenciaciones(
                year: $data['year'],
                assetId: $data['asset_id'],
                username: $data['auth_credential'],
                password: $data['second_auth_credential'] ?? null,
                serverUrl:$data['server_url']
            );


            $info = $importador->obtenerInfoAsset();
            return response()->json($info);
        }catch(\Exception $e){
            return response()->json([
                'success'=> false,
                'message'=> 'Error al obtener la información: ' . $e->getMessage()
            ], 500);
        }
    }

    public function getactivos(){
        return view('Georeferenciacion.activos');
    }

    public function datatablesactivos()
    {
        $GeoActivos = Georeferencing::with(['producer' => function($query) {
            $query->select('id_productor', 'grupo_trabajo');
        }])->select([
        'id',
        '_id',
        'start',
        'end',
        'today',

        'username',
        'deviceid',
        'phonenumber',
        'hora',
        'horas',
        'minutos',
        'segundos',
        'anos',
        'mes',
        'dias',

        'id_record',
        'promotor',
        'dni',
        'socio',
        'localidad',
        'municipio',
        'numero_parcela',
        'codigo_parcela',
        'nombre_parcela',
        'anio_incio_cultivo',
        'gps',
        'poligono',
        'area_gps',
        'area_gps_ha',

        'area_gps_display',
        'meta_instanceID',
        'meta_rootUuid',
        'meta_instanceName',

        'starttime',
        'endtime',
        'p_nombre',
        'ciclo_codigo',
        'subscriberid',
        'simid',
        'formhub_uuid',
        '__version__',
        '_xform_id_string',
        '_uuid',
        '_attachments',
        '_status',
        '_geolocation',

        '_submission_time',
        '_tags',
        '_notes',
        '_validation_status',
        '_submitted_by'
        ])->whereHas('parcela', function($query) {
            $query->whereHas('productor', function($q){
                $q->where('activo', 1);
            });
        })
        ->get()
        ->map(function($item) { //objeto
            $item->grupo_trabajo = $item->producer ? $item->producer->grupo_trabajo : null;
            return $item;
        });

        return response()->json($GeoActivos);
    }

    public function dataTables()
    {
        $Georef = Georeferencing::select([
        'id',
        '_id',
        'start',
        'end',
        'today',

        'username',
        'deviceid',
        'phonenumber',
        'hora',
        'horas',
        'minutos',
        'segundos',
        'anos',
        'mes',
        'dias',

        'id_record',
        'promotor',
        'dni',
        'socio',
        'localidad',
        'municipio',
        'numero_parcela',
        'codigo_parcela',
        'nombre_parcela',
        'anio_incio_cultivo',
        'gps',
        // 'poligono',
        'area_gps',
        'area_gps_ha',

        'area_gps_display',
        'meta_instanceID',
        'meta_rootUuid',
        'meta_instanceName',

        'starttime',
        'endtime',
        'p_nombre',
        'ciclo_codigo',
        'subscriberid',
        'simid',
        'formhub_uuid',
        '__version__',
        '_xform_id_string',
        '_uuid',
        '_attachments',
        '_status',
        '_geolocation',

        '_submission_time',
        '_tags',
        '_notes',
        '_validation_status',
        '_submitted_by'
        ])->get();

        // $Georef = Georeferencing::all();

        return response()->json($Georef);
    }
    public function exportar(Request $request)
    {
        $year = $request->get('year', date('Y'));
        
        $georeferencias = Georeferencing::where('__version__', 'LIKE', "%{$year}%")
            ->orWhere('today', 'LIKE', "%{$year}%")
            ->get();
        
        $filename = "georeferenciacion_{$year}_" . date('Y-m-d_H-i-s') . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv; charset=UTF-8',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
        ];

        $callback = function() use ($georeferencias) {
            $file = fopen('php://output', 'w');
            
            // BOM para UTF-8 (ayuda con caracteres especiales en Excel)
            fprintf($file, chr(0xEF).chr(0xBB).chr(0xBF));
            
            // Headers del CSV
            fputcsv($file, [
                'ID', 'ID Kobo', 'Start', 'End', 'Today', 'Username', 'Device ID',
                'Phone Number', 'Hora', 'Horas', 'Minutos', 'Segundos', 'Años',
                'Mes', 'Días', 'ID Record', 'Promotor', 'DNI', 'Socio',
                'Localidad', 'Municipio', 'Número Parcela', 'Código Parcela',
                'Nombre Parcela', 'Año Inicio Cultivo', 'GPS', 'Área GPS',
                'Área GPS (ha)', 'Área GPS Display', 'Instance ID', 'Root UUID',
                'Instance Name', 'Start Time', 'End Time', 'P Nombre',
                'Ciclo Código', 'Subscriber ID', 'SIM ID', 'Formhub UUID',
                'Versión', 'XForm ID', 'UUID', 'Attachments', 'Status',
                'Geolocalización', 'Submission Time', 'Tags', 'Notes',
                'Validation Status', 'Submitted By'
            ]);
            
            // Datos
            foreach ($georeferencias as $geo) {
                fputcsv($file, [
                    $geo->id,
                    $geo->_id,
                    $geo->start,
                    $geo->end,
                    $geo->today,
                    $geo->username,
                    $geo->deviceid,
                    $geo->phonenumber,
                    $geo->hora,
                    $geo->horas,
                    $geo->minutos,
                    $geo->segundos,
                    $geo->anos,
                    $geo->mes,
                    $geo->dias,
                    $geo->id_record,
                    $geo->promotor,
                    $geo->dni,
                    $geo->socio,
                    $geo->localidad,
                    $geo->municipio,
                    $geo->numero_parcela,
                    $geo->codigo_parcela,
                    $geo->nombre_parcela,
                    $geo->anio_incio_cultivo,
                    $geo->gps,
                    $geo->area_gps,
                    $geo->area_gps_ha,
                    $geo->area_gps_display,
                    $geo->meta_instanceID,
                    $geo->meta_rootUuid,
                    $geo->meta_instanceName,
                    $geo->starttime,
                    $geo->endtime,
                    $geo->p_nombre,
                    $geo->ciclo_codigo,
                    $geo->subscriberid,
                    $geo->simid,
                    $geo->formhub_uuid,
                    $geo->__version__,
                    $geo->_xform_id_string,
                    $geo->_uuid,
                    $geo->_attachments,
                    $geo->_status,
                    $geo->_geolocation,
                    $geo->_submission_time,
                    $geo->_tags,
                    $geo->_notes,
                    $geo->_validation_status,
                    $geo->_submitted_by
                ]);
            }
            
            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
    public function getParcelasSinPoligono()
    {
        $parcelasActivas = CoffeeParcelRegistry::whereHas('productor', function($q) {
            $q->where('activo', 1);
        })->pluck('id_parcela');

        $parcelasConPoligono = Georeferencing::whereHas('parcela', function($q) {
            $q->whereHas('productor', function($q2) {
                $q2->where('activo', 1);
            });
        })->pluck('codigo_parcela');

        $idsSinPoligono = $parcelasActivas->diff($parcelasConPoligono);

        // Traemos la info de esas parcelas
        $parcelasSin = CoffeeParcelRegistry::whereIn('id_parcela', $idsSinPoligono)
            ->with('productor:id,socio,localidad')
            ->get()
            ->map(function($parcela) {
                return [
                    'id_parcela' => $parcela->id_parcela,
                    'socio' => $parcela->socio ?? null,
                    'nombre_parcela' => $parcela->nombre_parcela ?? null,
                ];
            });

        return response()->json($parcelasSin);
    }


    public function indexmap()
    {
        return view('Georeferenciacion.mapa'); // Esta será tu vista principal del mapa
    }

    public function mapaformato(Request $request)
    {
        $searchIdParcela = trim($request->get('search_id_parcela', ''));
        
        // Obtener datos procesados (desde cache o API)
        $processedData = $this->getProcessedData();
        
        if (!$processedData) {
            return view('Georeferenciacion.mapaformato', [
                'message' => 'Error al cargar los datos base. Revise la conexión con KoboToolbox.',
                'defaultMapCenter' => [16.9202568, -92.5100693],
                'defaultMapZoom' => 13
            ]);
        }

        $selectedGeoRecord = null;
        $selectedParcelaRecord = null;
        $centerCoords = null;
        $polygonCoords = [];
        $message = null;
        $gpsDetails = null;

        if ($searchIdParcela) {
            // Búsqueda rápida usando el diccionario pre-procesado
            $selectedGeoRecord = $processedData['geo_by_codigo_parcela'][$searchIdParcela] ?? null;

            if ($selectedGeoRecord) {
                $gpsPointStr = $selectedGeoRecord['georeferenciacion/gps'] ?? null;
                $gpsDetails = $this->extractGpsDetails($gpsPointStr);
                
                if ($gpsDetails) {
                    $selectedGeoRecord['gps_latitud'] = number_format($gpsDetails['latitud'], 6);
                    $selectedGeoRecord['gps_longitud'] = number_format($gpsDetails['longitud'], 6);
                    $selectedGeoRecord['gps_altitud'] = number_format($gpsDetails['altitud'], 2) . ' m';
                    $selectedGeoRecord['gps_precision'] = number_format($gpsDetails['precision'], 2) . ' m';
                }
                
                $centerCoords = $this->parseGpsPoint($gpsPointStr);
                $polygonCoords = $this->parsePolygon($selectedGeoRecord['georeferenciacion/poligono'] ?? null);

                // Búsqueda rápida de la parcela correspondiente
                $selectedParcelaRecord = $processedData['parcela_by_id_parcela'][$searchIdParcela] ?? null;
                
                if ($selectedParcelaRecord) {
                    $poligonoDataStr = $selectedGeoRecord['georeferenciacion/poligono'] ?? null;
                    if ($poligonoDataStr) {
                        $selectedParcelaRecord['georeferenciacion/poligono'] = $poligonoDataStr;
                    }
                }

                // Búsqueda rápida del socio en el padrón
                $socioNombre = $selectedGeoRecord['georeferenciacion/socio'] ?? null;
                $selectedPadronRecord = $processedData['padron_by_socio'][$socioNombre] ?? null;
                
                if ($selectedPadronRecord) {
                    $selectedGeoRecord['productor_id'] = $selectedPadronRecord['id_productor'] ?? 'N/A';
                    $selectedGeoRecord['productor_region'] = $selectedPadronRecord['region'] ?? 'N/A';
                    $selectedGeoRecord['productor_grupo'] = $selectedPadronRecord['grupo_trabajo'] ?? 'N/A';
                    $selectedGeoRecord['productor_curp'] = $selectedPadronRecord['curp'] ?? 'N/A';
                    $selectedGeoRecord['productor_rfc'] = $selectedPadronRecord['rfc'] ?? 'N/A';
                    $selectedGeoRecord['productor_folio_cafetalero'] = $selectedPadronRecord['folio_cafetalero'] ?? 'N/A';
                    $selectedGeoRecord['productor_fecha_nacimiento'] = $selectedPadronRecord['fecha_nacimiento'] ?? 'N/A';
                    $selectedGeoRecord['productor_genero'] = ($selectedPadronRecord['genero'] ?? '') === 'H' ? 'Masculino' : 'Femenino';
                    $selectedGeoRecord['productor_estatus'] = $selectedPadronRecord['estatus'] ?? 'N/A';
                    $selectedGeoRecord['productor_ingreso'] = $selectedPadronRecord['ingreso'] ?? 'N/A';
                    $selectedGeoRecord['productor_num_parcelas'] = $selectedPadronRecord['numero_parcelas'] ?? 'Dato no encontrado en padrón';
                    $selectedGeoRecord['productor_activo'] = ($selectedPadronRecord['activo'] ?? '') === '1' ? 'Sí' : 'No';
                } else {
                    $selectedGeoRecord['productor_num_parcelas'] = "Socio no encontrado en padrón";
                    $message = "Parcela encontrada, pero el socio '$socioNombre' no fue hallado en el padrón. Verifique la consistencia de los datos.";
                }
            } else {
                $message = "No se encontró ninguna parcela con el ID: $searchIdParcela";
            }
        } else {
            $message = "Por favor, ingrese un ID de Parcela para buscar.";
        }

        $geoHeadersMap = [
            'georeferenciacion/municipio' => 'Municipio',
            'georeferenciacion/localidad' => 'Comunidad',
            'productor_grupo' => 'Grupo de Trabajo',
            'georeferenciacion/socio' => 'Socio',
            'productor_num_parcelas' => 'Número de parcelas',
            'georeferenciacion/codigo_parcela' => 'Código de la parcela',
            'georeferenciacion/nombre_parcela' => 'Nombre Parcela',
            'georeferenciacion/anio_incio_cultivo' => 'Año de inicio del cultivo',
            // 'georeferenciacion/area_gps_ha' => 'Superficie GPS (ha)',
            'gps_latitud' => 'Latitud',
            'gps_longitud' => 'Longitud',
            'gps_altitud' => 'Altitud',
            'gps_precision' => 'Precisión GPS',
            'productor_genero' => 'Género',
        ];

        $parcelaHeadersMap = [
            'superficie' => 'Superficie declarada (ha)',
            'norte' => 'Cultivo al norte',
            'norte_propietario' => 'Colindante norte',
            'sur' => 'Cultivo al sur',
            'sur_propietario' => 'Colindante sur',
            'este' => 'Cultivo al este',
            'este_propietario' => 'Colindante este',
            'oeste' => 'Cultivo oeste',
            'oeste_propietario' => 'Colindante oeste',
            'georeferenciacion/poligono' => 'Coordenadas del Polígono',
        ];

        return view('Georeferenciacion.mapaformato', [
            'geoRecord' => $selectedGeoRecord,
            'parcelaRecord' => $selectedParcelaRecord,
            'geoHeaders' => $geoHeadersMap,
            'parcelaHeaders' => $parcelaHeadersMap,
            'gpsDetails' => $gpsDetails,
            'centerCoords' => $centerCoords,
            'polygonCoords' => $polygonCoords,
            'defaultMapCenter' => [16.9202568, -92.5100693],
            'defaultMapZoom' => 14,
            'currentSearchId' => $searchIdParcela,
            'message' => $message
        ]);
    }

    public function getPoligonosformato(Request $request){
        $processedData = $this->getProcessedData();
        
        if (!$processedData) {
            return response()->json([
                'type' => 'FeatureCollection',
                'features' => []
            ]);
        }

        $features = [];
        
        foreach ($processedData['geo_by_codigo_parcela'] as $codigoParcela => $geoRecord) {
            $polygonCoords = $this->parsePolygon($geoRecord['georeferenciacion/poligono'] ?? null);
            
            if (empty($polygonCoords)) {
                continue; // Saltar si no hay polígono válido
            }

            // Convertir coordenadas al formato GeoJSON [lon, lat] (invertido)
            $coordinates = [array_map(function($coord) {
                return [$coord[1], $coord[0]]; // [lon, lat]
            }, $polygonCoords)];

            // Obtener datos del padrón para el grupo de trabajo
            $socioNombre = $geoRecord['georeferenciacion/socio'] ?? null;
            $padronRecord = $processedData['padron_by_socio'][$socioNombre] ?? null;

            $features[] = [
                'type' => 'Feature',
                'geometry' => [
                    'type' => 'Polygon',
                    'coordinates' => $coordinates
                ],
                'properties' => [
                    'codigo_parcela' => $codigoParcela,
                    'nombre_parcela' => $geoRecord['georeferenciacion/nombre_parcela'] ?? 'N/A',
                    'socio' => $socioNombre ?? 'N/A',
                    'localidad' => $geoRecord['georeferenciacion/localidad'] ?? 'N/A',
                    'municipio' => $geoRecord['georeferenciacion/municipio'] ?? 'N/A',
                    'grupo_trabajo' => $padronRecord['grupo_trabajo'] ?? null,
                    'area_gps_ha' => $geoRecord['georeferenciacion/area_gps_ha'] ?? 'N/A'
                ]
            ];
        }

        return response()->json([
            'type' => 'FeatureCollection',
            'features' => $features
        ]);
    }

    public function updateLocation(Request $request)
    {
        $recordId = $request->input('record_id');
        $searchIdParcela = $request->input('search_id_parcela');
        $newGpsCoords = $request->input('new_gps_coords');

        if (!$recordId || !$newGpsCoords) {
            return back()->with('error', 'Datos incompletos para actualizar la ubicación.');
        }

        try {
            // Construir la URL para el PATCH
            $url = $this->koboBaseUrl . '/api/v2/assets/' . $this->geoAssetUid . '/data/' . $recordId . '/';
            
            $response = Http::withHeaders([
                'Authorization' => 'Token ' . $this->koboApiToken,
                'Content-Type' => 'application/json',
            ])->patch($url, [
                'georeferenciacion/gps' => $newGpsCoords
            ]);

            if ($response->successful()) {
                // Invalidar cache para refrescar datos
                Cache::forget('processed_parcela_data');
                
                return redirect()
                    ->route('georeferenciacion.mapaformato', ['search_id_parcela' => $searchIdParcela])
                    ->with('success', "¡Ubicación de la parcela $searchIdParcela actualizada correctamente! Los datos se han refrescado.");
            } else {
                Log::error('Error al actualizar ubicación en Kobo', [
                    'status' => $response->status(),
                    'body' => $response->body()
                ]);
                
                return back()->with('error', 'Error al actualizar la ubicación en KoboToolbox. Código: ' . $response->status());
            }
        } catch (\Exception $e) {
            Log::error('Excepción al actualizar ubicación: ' . $e->getMessage());
            return back()->with('error', 'Error de conexión al actualizar la ubicación: ' . $e->getMessage());
        }
    }

    private function getProcessedData()
    {
        return Cache::remember('processed_parcela_data', $this->cacheDuration, function () {
            Log::info('--- Obteniendo datos frescos desde la API de KoboToolbox... ---');
            
            $geoApiUrl = $this->koboBaseUrl . '/api/v2/assets/' . $this->geoAssetUid . '/data/?format=json';
            $parcelaApiUrl = $this->koboBaseUrl . '/api/v2/assets/' . $this->parcelaAssetUid . '/data/?format=json';
            $padronApiUrl = $this->koboBaseUrl . '/api/v2/assets/' . $this->padronApiUid . '/data/?format=json';
            
            $geoDataResponse = $this->fetchKoboData($geoApiUrl);
            $parcelaDataResponse = $this->fetchKoboData($parcelaApiUrl);
            $padronDataResponse = $this->fetchKoboData($padronApiUrl);
            
            if (!$geoDataResponse || !$parcelaDataResponse || !$padronDataResponse) {
                Log::error('Error: No se pudieron obtener todos los datos de KoboToolbox');
                return null;
            }

            $geoRecords = $geoDataResponse['results'] ?? [];
            $parcelaRecords = $parcelaDataResponse['results'] ?? [];
            $padronRecords = $padronDataResponse['results'] ?? [];
            
            Log::info('Datos obtenidos exitosamente', [
                'geo_records' => count($geoRecords),
                'parcela_records' => count($parcelaRecords),
                'padron_records' => count($padronRecords)
            ]);
            
            return $this->processDataForLookup($geoRecords, $parcelaRecords, $padronRecords);
        });
    }
    private function processDataForLookup($geoRecords, $parcelaRecords, $padronRecords)
    {
        $geoByCodigoParcela = [];
        foreach ($geoRecords as $rec) {
            if (isset($rec['georeferenciacion/codigo_parcela'])) {
                $geoByCodigoParcela[$rec['georeferenciacion/codigo_parcela']] = $rec;
            }
        }

        $parcelaByIdParcela = [];
        foreach ($parcelaRecords as $rec) {
            if (isset($rec['id_parcela'])) {
                $parcelaByIdParcela[$rec['id_parcela']] = $rec;
            }
        }

        $padronBySocio = [];
        foreach ($padronRecords as $rec) {
            if (isset($rec['socio'])) {
                $padronBySocio[$rec['socio']] = $rec;
            }
        }

        return [
            'geo_by_codigo_parcela' => $geoByCodigoParcela,
            'parcela_by_id_parcela' => $parcelaByIdParcela,
            'padron_by_socio' => $padronBySocio
        ];
    }
    private function fetchKoboData($apiUrl, $timeout = 30)
    {
        try {
            $response = Http::timeout($timeout)
                ->withHeaders(['Authorization' => 'Token ' . $this->koboApiToken])
                ->get($apiUrl, ['limit' => 5000]);

            if ($response->successful()) {
                return $response->json();
            }

            Log::error("Error HTTP al obtener datos de Kobo", [
                'url' => $apiUrl,
                'status' => $response->status(),
                'body' => $response->body()
            ]);
            return null;
        } catch (\Exception $e) {
            Log::error("Excepción al obtener datos de Kobo", [
                'url' => $apiUrl,
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }
    private function parseGpsPoint($gpsString)
    {
        if (!$gpsString || !is_string($gpsString)) {
            return null;
        }

        $parts = explode(' ', $gpsString);
        if (count($parts) >= 2) {
            $lat = floatval($parts[0]);
            $lon = floatval($parts[1]);
            
            if ($lat >= -90 && $lat <= 90 && $lon >= -180 && $lon <= 180) {
                return [$lat, $lon];
            }
        }

        return null;
    }
    private function parsePolygon($polygonString)
    {
        if (!$polygonString || !is_string($polygonString)) {
            return [];
        }

        $coordinates = [];
        $points = explode(';', $polygonString);
        
        foreach ($points as $pointStr) {
            $parts = explode(' ', trim($pointStr));
            if (count($parts) >= 2) {
                $lat = floatval($parts[0]);
                $lon = floatval($parts[1]);
                
                if ($lat >= -90 && $lat <= 90 && $lon >= -180 && $lon <= 180) {
                    $coordinates[] = [$lat, $lon];
                }
            }
        }

        return $coordinates;
    }
    private function extractGpsDetails($gpsString)
    {
        if (!$gpsString || !is_string($gpsString)) {
            return null;
        }

        $parts = explode(' ', $gpsString);
        if (count($parts) >= 4) {
            return [
                'latitud' => floatval($parts[0]),
                'longitud' => floatval($parts[1]),
                'altitud' => floatval($parts[2]),
                'precision' => floatval($parts[3])
            ];
        }

        return null;
    }

    public function getPoligonos()
    {
        // $registros = \App\Models\Georeferencing::with('producer')->get();

        $registros = Georeferencing::whereHas('parcela', function($q) {
            $q->whereHas('productor', function($q2){
                $q2->where('activo', 1);
            });
        })->get();

        $features = [];

        foreach ($registros as $r) {
            if (!$r->poligono) continue;

            // Separar puntos
            $coords = explode(';', $r->poligono);
            $latlngs = [];

            foreach ($coords as $coord) {
                $parts = preg_split('/\s+/', trim($coord));
                if(count($parts) >= 2){
                    $lat = floatval($parts[0]);
                    $lng = floatval($parts[1]);
                    $latlngs[] = [$lng, $lat]; // GeoJSON usa [lng, lat]
                }
            }

            if(count($latlngs) > 2){
                $features[] = [
                    'type' => 'Feature',
                    'properties' => [
                        'codigo_parcela' => $r->codigo_parcela,
                        'nombre_parcela' => $r->nombre_parcela,
                        'socio' => $r->socio,
                        'localidad' => $r->localidad,
                        'grupo_trabajo' => $r->producer ? $r->producer->grupo_trabajo : null,
                    ],
                    'geometry' => [
                        'type' => 'Polygon',
                        'coordinates' => [$latlngs]
                    ]
                ];
            }
        }

        return response()->json([
            'type' => 'FeatureCollection',
            'features' => $features
        ]);
    }

    public function subirmapa(Request $request)
    {
        $file = $request->file('archivo');
        $extension = $file->getClientOriginalExtension();
        


        // Caso 1: JSON
        if ($extension === 'json') {
            $data = json_decode(file_get_contents($file), true);
            return response()->json($data);
        }

        // Caso 2: Excel
        if (in_array($extension, ['xls', 'xlsx'])) {
            $data = Excel::toArray([], $file)[0]; // primera hoja
            $puntos = collect($data)->skip(1)->map(function ($fila) {
                return [
                    'nombre' => $fila[0] ?? 'Sin nombre',
                    'lat' => (float) ($fila[1] ?? 0),
                    'lng' => (float) ($fila[2] ?? 0),
                ];
            })->filter(fn($p) => $p['lat'] && $p['lng']);

            return response()->json($puntos->values());
        }

        return response()->json(['error' => 'Formato no soportado'], 400);
    }



    public function sincronizarAutomatico()
    {
        $config = [
            'year' => date('Y'),
            'asset_id' => config('services.kobotoolbox.asset_id'),
            'username' => config('services.kobotoolbox.username'),
            'password' => config('services.kobotoolbox.password'),
            'server_url' => config('services.kobotoolbox.server_url', 'https://eu.kobotoolbox.org')
        ];

        if (!$config['username'] || !$config['password'] || !$config['asset_id']){
            Log::error('Configuración de KoboToolbox incompleta para sincronización automática');
            return response()->json([
                'success' => false,
                'message' => 'Configuración incompleta'
            ]);
        }

        try{
            $importador = new ImportadorGeoreferenciaciones(
                $config['year'],
                $config['username'],
                $config['password'],
                $config['asset_id'],
                $config['server_url']
            );

            $resultado = $importador->importar();

            Log::info('Sincronización automática completada', $resultado);
            return response()->json($resultado);

        }catch (\Exception $e){
            Log::error("Error en sincronización automática: {$e->getMessage()}");
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ]);
        }
    }
}
