<?php

namespace App\Services;

use App\Models\HumanRightsCase;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;

class ImportadorDerechoshumanos extends ImportadorBase
{
    protected string $modelClass = HumanRightsCase::class;
    protected string $idExterno = 'grupo_general/productor/dni';

    protected array $mapeoCampos = [
        '_id' => '_id',
        'formhub_uuid' => 'formhub/uuid',
        'start' => 'start',
        'end' => 'end',

        // Grupo General - Productor
        'nombre_encuestador' => 'grupo_general/productor/Nombre_del_encuestador',
        'dni' => 'grupo_general/productor/dni',
        'dni_o' => 'grupo_general/productor/dni_o',
        'socio' => 'grupo_general/productor/socio',
        'municipio' => 'grupo_general/productor/municipio',
        'localidad' => 'grupo_general/productor/localidad',
        'grupo_trabajo' => 'grupo_general/productor/grupo_trabajo',
        'telefono' => 'grupo_general/productor/telefono',
        'genero' => 'grupo_general/productor/genero',
        'fecha_nacimiento' => 'grupo_general/productor/fecha_nacimiento',

        // Grupo General - Mano de Obra
        'contrato_mano_obra' => 'grupo_general/mano_obra/contrato_mano_obra',
        'edad_jornaleros' => 'grupo_general/mano_obra/edad_jornaleros',
        'actividad_realizada' => 'grupo_general/mano_obra/actividad_realizada',
        'actividad_especifica' => 'grupo_general/mano_obra/actividad_especifica',
        'origen' => 'grupo_general/mano_obra/origen',
        'estado' => 'grupo_general/mano_obra/estado',
        'pais' => 'grupo_general/mano_obra/pais',
        'comunidad' => 'grupo_general/mano_obra/comunidad',
        'trabajo_forzado' => 'grupo_general/mano_obra/trabajo_forzado',
        'observaciones' => 'grupo_general/mano_obra/observaciones',

        // Firma
        'firma' => 'grupo_general/firma',

        // Metadatos
        '__version__' => '__version__',
        'instanceID' => 'meta/instanceID',
        'instanceName' => 'meta/instanceName',
        '_xform_id_string' => '_xform_id_string',
        '_uuid' => '_uuid',
        'rootUuid' => 'meta/rootUuid',
        '_attachments' => '_attachments',
        '_status' => '_status',
        '_geolocation' => '_geolocation',
        '_submission_time' => '_submission_time',
        '_tags' => '_tags',
        '_notes' => '_notes',
        '_validation_status' => '_validation_status',
        '_submitted_by' => '_submitted_by'
    ];

    public function __construct(
        int $year,
        string $assetId,
        ?string $username = null,
        ?string $password = null,
        string $serverUrl = 'https://eu.kobotoolbox.org'
    )
    {
        parent::__construct($year);

        $this->urlApi = "{$serverUrl}/api/v2/assets/{$assetId}/data/?format=json";

        $this->setCredenciales(
            $username ?: $this->username,
            $password ?: $this->password
        );

        $this->setHeaders([
            'Accept' => 'application/json',
            'Content-Type' => 'application/json'
        ]);
    }

    protected function extraerResultados(array $datos): array
    {
        return $datos['results'] ?? [];
    }

    protected function mapearDatos(array $datoApi): array
    {
        $datosMapeados = [];

        foreach ($this->mapeoCampos as $campoLocal => $campoApi) {
            $valor = $this->obtenerValorAnidado($datoApi, $campoApi);

            // Procesamiento especial para campos JSON (select_multiple)
            if (in_array($campoLocal, ['edad_jornaleros', 'actividad_realizada', '_attachments', '_geolocation', '_tags', '_notes'])) {
                if (is_string($valor)) {
                    // Si viene como string separado por espacios (común en select_multiple)
                    $valores = array_filter(explode(' ', trim($valor)));
                    $datosMapeados[$campoLocal] = !empty($valores) ? json_encode($valores) : null;
                } elseif (is_array($valor)) {
                    $datosMapeados[$campoLocal] = json_encode($valor);
                } else {
                    $datosMapeados[$campoLocal] = $valor;
                }
            } else {
                // Para el resto de campos, normalizar el valor
                $datosMapeados[$campoLocal] = $this->normalizarValor($valor);
            }
        }

        // Procesamiento especial para fechas
        $datosMapeados['start'] = $this->parsearFechaHora($datosMapeados['start'] ?? null);
        $datosMapeados['end'] = $this->parsearFechaHora($datosMapeados['end'] ?? null);
        $datosMapeados['fecha_nacimiento'] = $this->parsearFecha($datosMapeados['fecha_nacimiento'] ?? null);
        $datosMapeados['_submission_time'] = $this->parsearFechaHora($datosMapeados['_submission_time'] ?? null);

        return $datosMapeados;
    }

    protected function obtenerValorAnidado(array $datos, string $campo)
    {
        // Caso 1: la clave existe literal en el array
        if (array_key_exists($campo, $datos)) {
            return $datos[$campo];
        }

        $separadores = ['/', '.'];

        foreach ($separadores as $separador) {
            if (strpos($campo, $separador) !== false) {
                $claves = explode($separador, $campo);
                $valor = $datos;

                foreach ($claves as $clave) {
                    if (is_array($valor) && array_key_exists($clave, $valor)) {
                        $valor = $valor[$clave];
                    } else {
                        return null;
                    }
                }
                return $valor;
            }
        }

        // Si no hay separadores, es un campo simple
        return $datos[$campo] ?? null;
    }

    protected function parsearFecha(?string $fecha): ?string
    {
        if (!$fecha) return null;

        try {
            $timestamp = strtotime($fecha);
            return $timestamp ? date('Y-m-d', $timestamp) : null;
        } catch (\Exception $e) {
            Log::warning("Error al parsear fecha: {$fecha}", ['exception' => $e->getMessage()]);
            return null;
        }
    }

    protected function parsearFechaHora(?string $fechaHora): ?string
    {
        if (!$fechaHora) return null;

        try {
            $timestamp = strtotime($fechaHora);
            return $timestamp ? date('Y-m-d H:i:s', $timestamp) : null;
        } catch (\Exception $e) {
            Log::warning("Error al parsear fecha-hora: {$fechaHora}", ['exception' => $e->getMessage()]);
            return null;
        }
    }

    public function obtenerInfoAsset(): array
    {
        try {
            $assetUrl = str_replace('/data/?format=json', '', $this->urlApi);
            
            $httpClient = Http::timeout(30);

            if ($this->username && $this->password) {
                $httpClient = $httpClient->withBasicAuth($this->username, $this->password);
            }
            
            if (!empty($this->headers)) {
                $httpClient = $httpClient->withHeaders($this->headers);
            }
            
            $response = $httpClient->get($assetUrl);
            
            if (!$response->successful()) {
                return [
                    'success' => false, 
                    'message' => $this->getErrorMessage($response->status())
                ];
            }
            
            $data = $response->json();
            return [
                'success' => true,
                'name' => $data['name'] ?? 'Sin nombre',
                'deployment_count' => $data['deployment__submission_count'] ?? 0,
                'date_modified' => $data['date_modified'] ?? null,
                'asset_type' => $data['asset_type'] ?? 'survey'
            ];
            
        } catch (\Exception $e) {
            Log::error("Error al obtener info del asset: {$e->getMessage()}");
            return [
                'success' => false, 
                'message' => 'Error al obtener información: ' . $e->getMessage()
            ];
        }
    }

    protected function procesarRegistro(array $datoApi)
    {
        $datosMapeados = $this->mapearDatos($datoApi);
        $datosMapeados['version'] = $this->year;

        // Obtener el DNI del productor
        $valorIdExterno = $this->obtenerValorAnidado($datoApi, 'grupo_general/productor/dni');

        // Si no hay DNI, generar uno temporal
        if (empty($valorIdExterno)) {
            $valorIdExterno = 'SIN_DNI_' . ($datoApi['_id'] ?? uniqid());
            $datosMapeados['dni'] = $valorIdExterno;
            
            Log::warning("Registro sin DNI, asignando: {$valorIdExterno}", [
                '_id' => $datoApi['_id'] ?? null
            ]);
        }

        // Buscar registro existente por DNI y versión
        $model = $this->modelClass::where('dni', $valorIdExterno)
            ->where('version', $this->year)
            ->first();

        if ($model) {
            // Actualizar registro existente
            $model->update($datosMapeados);
            Log::info("Actualizado registro con DNI: {$valorIdExterno}");
        } else {
            // Crear nuevo registro
            try {
                $this->modelClass::create($datosMapeados);
                Log::info("Creado nuevo registro con DNI: {$valorIdExterno}");
            } catch (\Exception $e) {
                Log::error('Error al insertar registro', [
                    '_id' => $datoApi['_id'] ?? null,
                    'dni' => $valorIdExterno,
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);
            }
        }
    }

    protected function validarProductor($dni): bool
    {
        if (empty($dni)) {
            Log::warning("DNI vacío, no se puede validar");
            return false;
        }
        
        // Verificar que el productor existe en la base de datos
        return \App\Models\Producer::where('curp', $dni)
            ->orWhere('dni', $dni)
            ->exists();
    }
    
    public function validarCredenciales(): bool
    {
        $info = $this->obtenerInfoAsset();
        return $info['success'] ?? false;
    }

    protected function normalizarValor($valor)
    {
        if ($valor === null) {
            return null;
        }

        // Si es array, convertir a JSON
        if (is_array($valor)) {
            return json_encode($valor, JSON_UNESCAPED_UNICODE);
        }

        // Si es collection, convertir a JSON
        if ($valor instanceof \Illuminate\Support\Collection) {
            return json_encode($valor->all(), JSON_UNESCAPED_UNICODE);
        }

        // Si es objeto, intentar convertir a string o JSON
        if (is_object($valor)) {
            if (method_exists($valor, '__toString')) {
                return (string) $valor;
            }
            return json_encode($valor, JSON_UNESCAPED_UNICODE);
        }

        // Si es booleano, convertir a string
        if (is_bool($valor)) {
            return $valor ? '1' : '0';
        }

        // Para escalares (string, int, float), devolver tal como están
        return $valor;
    }

    protected function getErrorMessage(int $statusCode): string
    {
        return match($statusCode) {
            401 => 'Credenciales inválidas. Verifica tu usuario y contraseña.',
            403 => 'Acceso denegado. No tienes permisos para acceder a este formulario.',
            404 => 'Formulario no encontrado. Verifica el Asset ID.',
            500 => 'Error del servidor de KoBoToolbox. Intenta más tarde.',
            default => "Error HTTP {$statusCode} al conectar con KoBoToolbox."
        };
    }
}